# This is just the code to illustrate the experimental setup through Figure 4

import numpy as np
import matplotlib.pyplot as plt
from mpl_toolkits.mplot3d import Axes3D

# Parameters
theta_deg = 45  # half-angle of the cap in degrees
theta = np.deg2rad(theta_deg)
resolution = 60  # resolution for the grid

# Create spherical coordinates grid
phi = np.linspace(0, np.pi, resolution)
psi = np.linspace(0, 2*np.pi, resolution)
phi_grid, psi_grid = np.meshgrid(phi, psi)

# Convert to Cartesian coordinates for the full sphere
x_sphere = np.sin(phi_grid) * np.cos(psi_grid)
y_sphere = np.sin(phi_grid) * np.sin(psi_grid)
z_sphere = np.cos(phi_grid)

# Mask for the cap region
x_cap = np.where(phi_grid <= theta, x_sphere, np.nan)
y_cap = np.where(phi_grid <= theta, y_sphere, np.nan)
z_cap = np.where(phi_grid <= theta, z_sphere, np.nan)

# Edge vector on the cap boundary in the XZ-plane
v_edge = np.array([np.sin(theta), 0, np.cos(theta)])
# Midpoint vector for angle annotation
v_mid = np.array([np.sin(theta/2), 0, np.cos(theta/2)])

# Create arc points for angle phi
t = np.linspace(0, theta, 100)
arc_x = np.sin(t)
arc_y = np.zeros_like(t)
arc_z = np.cos(t)

# Plot
fig = plt.figure(figsize=(6, 6), facecolor='white')
ax = fig.add_subplot(111, projection='3d', facecolor='white')

# Sphere wireframe
ax.plot_wireframe(x_sphere, y_sphere, z_sphere, color='#BBBBBB', alpha=0.4, linewidth=0.6)

# Spherical cap surface (more opaque)
ax.plot_surface(x_cap, y_cap, z_cap, color='#0173b2', alpha=0.4, linewidth=0, rcount=resolution, ccount=resolution)

# Central axis vector v0
ax.quiver(0, 0, 0, 0, 0, 1, color='#d62728', linewidth=2.5, arrow_length_ratio=0.1)
ax.text(0, 0, 1.08, '$v_0$', color='#d62728', fontsize=12, ha='center')

# Edge vector from origin to cap boundary
ax.quiver(0, 0, 0, *v_edge, color='#2ca02c', linewidth=2.5, arrow_length_ratio=0.1)
ax.text(*(1.08 * v_edge), '$v_{edge}$', color='#2ca02c', fontsize=12, ha='left')

# Angle arc and annotation φ between v0 and v_edge
ax.plot(arc_x, arc_y, arc_z, color='black', linestyle='-', linewidth=1.5)
ax.text(*(1.05 * v_mid), r'$\phi$', color='black', fontsize=14, ha='center')

# Axes limits and ticks (reduced density)
ax.set_xlim([-1, 1]); ax.set_ylim([-1, 1]); ax.set_zlim([-1, 1])
ax.set_xticks([-1, 0, 1])
ax.set_yticks([-1, 0, 1])
ax.set_zticks([-1, 0, 1])

# Labels and title
ax.set_xlabel('X', labelpad=10)
ax.set_ylabel('Y', labelpad=10)
ax.set_zlabel('Z', labelpad=10)
# ax.set_title('Spherical Cap Sampling Region', pad=20, fontsize=14)

plt.tight_layout()
plt.show()
